<?php
/* --------------------------------------------------------------
  ImageHandler.inc.php 2019-06-07
  Gambio GmbH
  http://www.gambio.de
  Copyright (c) 2019 Gambio GmbH
  Released under the GNU General Public License (Version 2)
  [http://www.gnu.org/licenses/gpl-2.0.html]
  --------------------------------------------------------------*/

namespace StyleEdit;

/**
 * Class ImageHandler
 *
 * Will handle all the image operations of StyleEdit.
 *
 * @package StyleEdit
 */
class ImageHandler
{
	/**
	 * @var string
	 */
	protected $imageDirectoryPath;


	/**
	 * ImageHandler constructor.
	 *
	 * @param string $p_imageDirectoryPath The path to the image directory.
	 *
	 * @throws \InvalidArgumentException If the image directory is not writable.
	 */
	public function __construct($p_imageDirectoryPath)
	{
		if(!file_exists($p_imageDirectoryPath))
		{
			@mkdir($p_imageDirectoryPath, 0777, true); 
			@chmod($p_imageDirectoryPath, 0777); 
		}
		
		if(!is_writable($p_imageDirectoryPath))
		{
			throw new \InvalidArgumentException('The provided image directory is not writable: ' . $p_imageDirectoryPath);
		}
		
		$this->imageDirectoryPath = (string)$p_imageDirectoryPath;
	}


	/**
	 * Get image file names in an array.
	 *
	 * @return array
	 */
	public function getImageFilenames()
	{
		$images = array();

		foreach(glob($this->imageDirectoryPath . '/*') as $filename)
		{
			if($filename === '.' || $filename === '..'
			   || preg_match('/.png|.PNG|.jpg|.JPG|.jpeg|.JPEG|.bmp|.BMP|.gif|.GIF/', $filename) === 0
			)
			{
				continue;
			}

			$images[] = Utf8Converter::basename($filename);
		}

		return $images;
	}


	/**
	 * Move a new image file to the images directory.
	 *
	 * @param array $file The entry that is available on the $_FILES array.
	 *
	 * @return string Returns the filename of the new image.
	 */
	public function move(array $file)
	{
		$basename = Utf8Converter::basename($file['name']);
		
		move_uploaded_file($file['tmp_name'], $this->imageDirectoryPath . '/' . preg_replace("/[^A-Za-z0-9\_\-\.]/", '-', $basename));
		
		return $basename;
	}
	
	/**
	 * Copy a new image file to the images directory.
	 *
	 * @param array $file The entry that is available on the $_FILES array.
	 *
	 * @return string Returns the filename of the new image.
	 */
	public function copy(array $file)
	{
		$basename = Utf8Converter::basename($file['name']);
		
		copy($file['tmp_name'], $this->imageDirectoryPath . preg_replace("/[^A-Za-z0-9\_\-\.]/", '-', $basename));
		
		return $basename;
	}


	/**
	 * Delete an existing file name.
	 *
	 * @param string $p_filename Provide just the filename and not the whole path.
	 */
	public function delete($p_filename)
	{
		$path = $this->imageDirectoryPath . Utf8Converter::basename((string)$p_filename);

		if(file_exists($path) && is_writable($path))
		{
			unlink($path);
		}
	}
}